/**
 * API route for triggering FindAll event discovery
 * This calls the Parallel FindAll API to discover new events
 */

import { NextResponse } from 'next/server';
import { 
  createFindAllRun, 
  getRunStatus, 
  getRunResult, 
  transformCandidateToEvent 
} from '@/lib/parallel-api';
import { addEvents } from '@/lib/events-store';

export async function POST() {
  try {
    // Get the current date and calculate date range
    const today = new Date();
    const pastMonth = new Date(today);
    pastMonth.setMonth(pastMonth.getMonth() - 1);
    
    const futureMonths = new Date(today);
    futureMonths.setMonth(futureMonths.getMonth() + 6);

    const objective = `FindAll major B2B and B2C events happening in the Middle East region (UAE, Saudi Arabia, Qatar, Bahrain, Kuwait, Oman, Jordan, Egypt, Israel, Lebanon) from ${pastMonth.toISOString().split('T')[0]} to ${futureMonths.toISOString().split('T')[0]} with at least 1000 expected attendees. Include conferences, exhibitions, summits, festivals, trade shows, concerts, networking events, and sports events.`;

    const matchConditions = [
      {
        name: 'middle_east_location',
        description: 'Event must be held in a Middle East country (UAE, Saudi Arabia, Qatar, Bahrain, Kuwait, Oman, Jordan, Egypt, Israel, Lebanon)'
      },
      {
        name: 'minimum_attendees',
        description: 'Event must have at least 1000 expected or confirmed attendees'
      },
      {
        name: 'valid_event_type',
        description: 'Event must be a B2B or B2C event such as conference, exhibition, summit, festival, trade show, concert, sports event, or networking event'
      }
    ];

    const enrichments = [
      { name: 'event_name', description: 'Official name of the event suitable for display' },
      { name: 'event_date', description: 'Start date of the event in YYYY-MM-DD format' },
      { name: 'event_end_date', description: 'End date of the event in YYYY-MM-DD format if multi-day' },
      { name: 'attendees_range', description: 'Expected number of attendees as a range (e.g., "5000-10000")' },
      { name: 'venue_name', description: 'Name of the venue where the event is held' },
      { name: 'city', description: 'City where the event is held' },
      { name: 'country', description: 'Country where the event is held' },
      { name: 'ticket_available', description: 'Whether tickets are currently available (yes/no)' },
      { name: 'ticket_pricing', description: 'Ticket pricing information if available' },
      { name: 'one_liner_description', description: 'One sentence description of the event' },
      { name: 'attendee_profiles', description: 'Description of typical attendees (e.g., "Tech executives, startups, investors")' },
      { name: 'sponsors', description: 'List of major sponsors if any, comma separated' },
      { name: 'event_type', description: 'Type of event: conference, exhibition, summit, festival, trade-show, concert, sports, networking, workshop, or other' },
      { name: 'website_url', description: 'Official website URL of the event' }
    ];

    // Create the FindAll run
    const { findall_id } = await createFindAllRun({
      objective,
      entityType: 'events',
      matchConditions,
      enrichments,
      generator: 'pro',
      matchLimit: 50,
    });

    return NextResponse.json({
      success: true,
      findall_id,
      message: 'Discovery started. Poll /api/discover/status for updates.',
    });
  } catch (error) {
    console.error('Error starting discovery:', error);
    return NextResponse.json(
      { error: 'Failed to start event discovery', details: String(error) },
      { status: 500 }
    );
  }
}

/**
 * GET endpoint to check status and retrieve results
 */
export async function GET(request: Request) {
  try {
    const { searchParams } = new URL(request.url);
    const findallId = searchParams.get('findall_id');

    if (!findallId) {
      return NextResponse.json(
        { error: 'findall_id is required' },
        { status: 400 }
      );
    }

    // Get status
    const status = await getRunStatus(findallId);

    // If completed, get results and transform
    if (!status.status.is_active && status.status.status === 'completed') {
      const result = await getRunResult(findallId);
      
      // Transform candidates to events
      const events = result.candidates
        .filter(c => c.match_status === 'matched')
        .map((candidate, index) => transformCandidateToEvent(candidate, index));

      // Add to store
      addEvents(events);

      return NextResponse.json({
        status: 'completed',
        findall_id: findallId,
        metrics: status.status.metrics,
        events_added: events.length,
      });
    }

    return NextResponse.json({
      status: status.status.status,
      findall_id: findallId,
      is_active: status.status.is_active,
      metrics: status.status.metrics,
    });
  } catch (error) {
    console.error('Error checking discovery status:', error);
    return NextResponse.json(
      { error: 'Failed to check discovery status', details: String(error) },
      { status: 500 }
    );
  }
}

